/*
 * main.c
 *
 *  Created on: Aug 30, 2015
 *      Author: peter
 */

#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/io.h>
#include <stdint.h>



#define	BLINK_FAST			100
#define	BLINK_SLOW			500
#define	IGNITER_THRESHOLD	64

#define	SIGNAL_SET		PORTC |= _BV(PC5)
#define SIGNAL_CLR		PORTC &= ~_BV(PC5)

#define	IGNITE_SET		PORTC |= _BV(PC2)
#define	IGNITE_CLR		PORTC &= ~_BV(PC2)

#define	DELAY_BIT0		(PIND & _BV(PD3))
#define	DELAY_BIT1		(PIND & _BV(PD4))
#define	DELAY_BIT2		(PIND & _BV(PD5))
#define	DELAY_BIT3		(PIND & _BV(PD6))
#define	DELAY_BIT4		(PIND & _BV(PD7))

#define	IGNITE_BIT0		(PINB & _BV(PB0))
#define	IGNITE_BIT1		(PINB & _BV(PB1))

#define	G_SENSOR_BIT	(PIND & _BV(PD2))

#define	IN_PIN			(PINB & _BV(PB2))



volatile uint8_t  igniterConnected = 0;
volatile uint8_t  sensorPressed    = 0;
volatile uint16_t delayTime        = 0;



ISR (INT0_vect){
	sensorPressed = 1;
}



ISR (ADC_vect){
	igniterConnected = ADC > IGNITER_THRESHOLD ? 1 : 0;
}



ISR (TIMER1_COMPA_vect){
	if(delayTime)
		--delayTime;
}



void delay(uint16_t dt){ // in 10th of second
	TCNT1     = 0;
	delayTime = dt;
	while(delayTime)
		;
}



uint8_t getIgnitionTime(){
	const uint8_t INDEX =
			(IGNITE_BIT0 ? 0 : 1) |
			(IGNITE_BIT1 ? 0 : 2) ;

	// ignition times in 1/10 of second
	static const uint8_t IGNITE_DATA[4] = { 1, 5, 10, 20 };

	return IGNITE_DATA[INDEX];
}



uint16_t getDelayTime(){
	const uint16_t DELAY_TIME =
			(DELAY_BIT0 ? 0 : 0x01) |
			(DELAY_BIT1 ? 0 : 0x02) |
			(DELAY_BIT2 ? 0 : 0x04) |
			(DELAY_BIT3 ? 0 : 0x08) |
			(DELAY_BIT4 ? 0 : 0x10) ;

	return 10*(DELAY_TIME ? DELAY_TIME : 1); // minimum is 1 second
}



void init(){
	DDRB = 0x00;
	DDRC = _BV(PC2) | _BV(PC5);
	DDRD = 0x00;

	PORTB = _BV(PB0) | _BV(PB1) | _BV(PB2);
	PORTC = 0x00;
	PORTD = _BV(PD2) | _BV(PD3) | _BV(PD4) | _BV(PD5) | _BV(PD6) | _BV(PD7);

	// G sensor button interrupt
	MCUCR |= _BV(ISC01);
	GICR  |= _BV(INT0);

	// continuous AD conversion (ignitor detection)
	ADMUX  = 0x03 | _BV(REFS0) | _BV(REFS1);
	ADCSRA = _BV(ADEN) | _BV(ADSC) | _BV(ADFR) | _BV(ADPS2) | _BV(ADIF) | _BV(ADIE);

	// 0.1s delay timer
	TCCR1A = 0x00;
	TCCR1B = _BV(WGM12) | _BV(CS11);
	OCR1A  = 12500; // FOSC(1 000 000) / prescaler(8) / 10
	TIMSK  = _BV(OCIE1A);

	sei();
}



int main(){
	init();

	while(1){
		// wait for rocket start
		sensorPressed = 0;
		while(1){
			SIGNAL_CLR;
			_delay_ms(BLINK_FAST);
			if(sensorPressed)
				break;
			if(!igniterConnected)
				SIGNAL_SET;
			_delay_ms(BLINK_FAST);
			if(sensorPressed)
				break;
		}

		// timer delay
		SIGNAL_SET;
		delay(getDelayTime());

		// ignite
		IGNITE_SET;
		delay(getIgnitionTime());

		// shut down
		IGNITE_CLR;

		// wait for reset cycle
		while(1){
			SIGNAL_CLR;
			_delay_ms(BLINK_SLOW);
			if(!IN_PIN)
				break;
				SIGNAL_SET;
			_delay_ms(BLINK_SLOW);
			if(!IN_PIN)
				break;
		}
		SIGNAL_CLR;
	}

	return 0;
}
